/* Sales Monitor Pro - Dashboard JS */
(function ($) {
    'use strict';

    /* ─── State ─── */
    let monthChart = null;
    let currentYear  = new Date().getFullYear();
    let currentMonth = new Date().getMonth() + 1;
    const MONTHS_HE  = ['','ינואר','פברואר','מרץ','אפריל','מאי','יוני','יולי','אוגוסט','ספטמבר','אוקטובר','נובמבר','דצמבר'];
    const currency   = smpData.currency || '₪';

    /* ─── Init ─── */
    $(document).ready(function () {
        loadTodayStats();
        loadMonthStats();
        loadRecentSales();
        loadLiveVisitors();
        initMonthNav();
        initDayPicker();
        initPushButtons();

        // Auto-refresh every 30 seconds
        setInterval(() => {
            loadTodayStats();
            loadLiveVisitors();
        }, 30000);
    });

    /* ─── Today Stats ─── */
    function loadTodayStats() {
        fetch(smpData.apiUrl + 'sales/today', {
            headers: { 'X-WP-Nonce': smpData.nonce }
        })
        .then(r => r.json())
        .then(data => {
            animateNumber('#kpi-today-orders', data.orders);
            animateNumber('#kpi-today-revenue', data.revenue, currency);
        });
    }

    /* ─── Month Stats ─── */
    function loadMonthStats(year, month) {
        year  = year  || currentYear;
        month = month || currentMonth;

        $('#smp-current-month').text(MONTHS_HE[month] + ' ' + year);

        fetch(`${smpData.apiUrl}sales/month?year=${year}&month=${month}`, {
            headers: { 'X-WP-Nonce': smpData.nonce }
        })
        .then(r => r.json())
        .then(data => {
            animateNumber('#kpi-month-orders',  data.total_orders);
            animateNumber('#kpi-month-revenue', data.total_revenue, currency);
            renderMonthChart(data.days, year, month);
        });
    }

    /* ─── Monthly Chart ─── */
    function renderMonthChart(days, year, month) {
        const daysInMonth = new Date(year, month, 0).getDate();
        const labels   = [];
        const orders   = [];
        const revenues = [];

        // Fill all days
        for (let d = 1; d <= daysInMonth; d++) {
            const dateStr = `${year}-${String(month).padStart(2,'0')}-${String(d).padStart(2,'0')}`;
            labels.push(d);
            const found = (days || []).find(x => x.day === dateStr);
            orders.push(found ? parseInt(found.orders) : 0);
            revenues.push(found ? parseFloat(found.revenue) : 0);
        }

        const ctx = document.getElementById('smp-monthly-chart');
        if (!ctx) return;

        if (monthChart) monthChart.destroy();

        monthChart = new Chart(ctx, {
            type: 'bar',
            data: {
                labels,
                datasets: [
                    {
                        label: 'הזמנות',
                        data: orders,
                        backgroundColor: 'rgba(99, 102, 241, 0.8)',
                        borderRadius: 6,
                        yAxisID: 'y',
                    },
                    {
                        label: 'הכנסות (₪)',
                        data: revenues,
                        backgroundColor: 'rgba(16, 185, 129, 0.8)',
                        borderRadius: 6,
                        yAxisID: 'y1',
                        type: 'line',
                        borderColor: 'rgba(16, 185, 129, 1)',
                        borderWidth: 2,
                        fill: false,
                        tension: 0.4,
                        pointRadius: 3,
                    }
                ]
            },
            options: {
                responsive: true,
                interaction: { mode: 'index' },
                plugins: {
                    legend: { position: 'top', rtl: true },
                    tooltip: {
                        rtl: true,
                        callbacks: {
                            label: ctx => ctx.datasetIndex === 1
                                ? `${currency}${ctx.raw.toLocaleString()}`
                                : `${ctx.raw} הזמנות`
                        }
                    }
                },
                scales: {
                    y: {
                        type: 'linear',
                        position: 'right',
                        ticks: { stepSize: 1 },
                        title: { display: true, text: 'הזמנות' }
                    },
                    y1: {
                        type: 'linear',
                        position: 'left',
                        grid: { drawOnChartArea: false },
                        title: { display: true, text: 'הכנסות' }
                    }
                }
            }
        });
    }

    /* ─── Month Nav ─── */
    function initMonthNav() {
        $('#smp-prev-month').on('click', () => {
            currentMonth--;
            if (currentMonth < 1) { currentMonth = 12; currentYear--; }
            loadMonthStats(currentYear, currentMonth);
        });
        $('#smp-next-month').on('click', () => {
            const now = new Date();
            if (currentYear >= now.getFullYear() && currentMonth >= now.getMonth()+1) return;
            currentMonth++;
            if (currentMonth > 12) { currentMonth = 1; currentYear++; }
            loadMonthStats(currentYear, currentMonth);
        });
    }

    /* ─── Day Picker ─── */
    function initDayPicker() {
        $('#smp-search-day').on('click', () => {
            const date = $('#smp-date-picker').val();
            if (!date) return;
            fetch(`${smpData.apiUrl}sales/day/${date}`, {
                headers: { 'X-WP-Nonce': smpData.nonce }
            })
            .then(r => r.json())
            .then(data => {
                $('#day-orders').text(data.orders);
                $('#day-revenue').text(`${currency}${parseFloat(data.revenue).toLocaleString()}`);
                $('#smp-day-result').slideDown();
            });
        });
    }

    /* ─── Recent Sales ─── */
    function loadRecentSales() {
        fetch(smpData.apiUrl + 'sales/recent?limit=10', {
            headers: { 'X-WP-Nonce': smpData.nonce }
        })
        .then(r => r.json())
        .then(orders => {
            const tbody = $('#smp-recent-sales');
            tbody.empty();
            if (!orders || orders.length === 0) {
                tbody.append('<tr><td colspan="5" style="text-align:center;padding:20px;color:#888">אין הזמנות עדיין</td></tr>');
                return;
            }
            orders.forEach(order => {
                const date = new Date(order.post_date).toLocaleDateString('he-IL');
                const name = `${order.billing_first||''} ${order.billing_last||''}`.trim() || 'לקוח';
                const total = `${currency}${parseFloat(order.total||0).toLocaleString()}`;
                tbody.append(`
                    <tr>
                        <td><strong>#${order.order_id}</strong></td>
                        <td>${date}</td>
                        <td>${name}</td>
                        <td class="smp-revenue-cell">${total}</td>
                        <td><a href="${smpData.ajaxUrl.replace('admin-ajax.php','')}post.php?post=${order.order_id}&action=edit" class="smp-btn-tiny" target="_blank">צפה</a></td>
                    </tr>
                `);
            });
        });
    }

    /* ─── Live Visitors ─── */
    function loadLiveVisitors() {
        fetch(smpData.apiUrl + 'visitors/live', {
            headers: { 'X-WP-Nonce': smpData.nonce }
        })
        .then(r => r.json())
        .then(data => {
            // Count
            $('#smp-visitor-count').text(data.count);
            $('#smp-last-refresh').text('עודכן: ' + new Date().toLocaleTimeString('he-IL'));

            // Channels
            const channelColors = {
                'חיפוש אורגני': '#4285F4',
                'Facebook':     '#1877F2',
                'Instagram':    '#E1306C',
                'Google Ads':   '#FBBC04',
                'ישיר':         '#34A853',
                'TikTok':       '#000000',
                'WhatsApp':     '#25D366',
                'default':      '#6366F1',
            };
            const grid = $('#smp-channels-grid');
            grid.empty();
            (data.channels || []).forEach(ch => {
                const color = channelColors[ch.channel] || channelColors.default;
                grid.append(`
                    <div class="smp-channel-chip" style="border-color:${color}">
                        <span class="smp-channel-dot" style="background:${color}"></span>
                        <strong>${ch.channel}</strong>
                        <span class="smp-channel-count">${ch.cnt}</span>
                    </div>
                `);
            });
            if (!data.channels || data.channels.length === 0) {
                grid.append('<div class="smp-no-data">אין גולשים פעילים כרגע</div>');
            }

            // Visitors list
            const list = $('#smp-visitors-list');
            list.empty();
            (data.visitors || []).slice(0, 8).forEach(v => {
                const timeAgo = getTimeAgo(v.last_seen);
                const page = v.page_title || v.page_url || 'דף לא ידוע';
                list.append(`
                    <div class="smp-visitor-row">
                        <div class="smp-visitor-icon">👤</div>
                        <div class="smp-visitor-info">
                            <div class="smp-visitor-page">${page}</div>
                            <div class="smp-visitor-meta">${v.channel} · ${timeAgo}</div>
                        </div>
                    </div>
                `);
            });
        });
    }

    /* ─── Push Buttons ─── */
    function initPushButtons() {
        $('#smp-enable-push').on('click', () => {
            if ('Notification' in window && 'serviceWorker' in navigator) {
                Notification.requestPermission().then(perm => {
                    if (perm === 'granted') {
                        $('#smp-push-result').html('<span style="color:green">✅ התראות הופעלו!</span>');
                    }
                });
            }
        });

        $('#smp-test-push').on('click', function () {
            $(this).text('שולח...').prop('disabled', true);
            // Test via WP Admin - trigger a dummy notification
            $.ajax({
                url: smpData.ajaxUrl,
                method: 'POST',
                data: {
                    action: 'smp_test_notification',
                    nonce:  smpData.nonce,
                },
                success: (res) => {
                    $('#smp-push-result').html('<span style="color:green">✅ ההתראה נשלחה!</span>');
                    setTimeout(() => { $('#smp-push-result').empty(); }, 3000);
                },
                complete: () => {
                    $('#smp-test-push').text('שלח התראת בדיקה 🔔').prop('disabled', false);
                }
            });
        });
    }

    /* ─── Helpers ─── */
    function animateNumber(selector, target, prefix) {
        const $el = $(selector);
        const start = parseFloat($el.data('val') || 0);
        const end   = parseFloat(target || 0);
        $el.data('val', end);
        $({ val: start }).animate({ val: end }, {
            duration: 800,
            easing: 'swing',
            step(now) {
                const formatted = now >= 1000
                    ? now.toLocaleString('he-IL', { maximumFractionDigits: 0 })
                    : Math.round(now).toLocaleString('he-IL');
                $el.text(prefix ? prefix + formatted : formatted);
            },
            complete() {
                const formatted = end >= 1000
                    ? end.toLocaleString('he-IL', { maximumFractionDigits: 0 })
                    : end.toLocaleString('he-IL');
                $el.text(prefix ? prefix + formatted : formatted);
            }
        });
    }

    function getTimeAgo(dateStr) {
        const diff = Math.floor((Date.now() - new Date(dateStr).getTime()) / 1000);
        if (diff < 60)  return `לפני ${diff} שניות`;
        if (diff < 3600) return `לפני ${Math.floor(diff/60)} דקות`;
        return `לפני ${Math.floor(diff/3600)} שעות`;
    }

})(jQuery);
