<?php
if ( ! defined( 'ABSPATH' ) ) exit;

class SMP_Database {

    public static function install() {
        global $wpdb;
        $charset = $wpdb->get_charset_collate();

        $sql = [];

        /* גולשים פעילים */
        $sql[] = "CREATE TABLE IF NOT EXISTS {$wpdb->prefix}smp_visitors (
            id           BIGINT UNSIGNED AUTO_INCREMENT PRIMARY KEY,
            session_id   VARCHAR(64)  NOT NULL,
            ip           VARCHAR(45)  NOT NULL,
            page_url     TEXT,
            page_title   VARCHAR(255),
            referrer     TEXT,
            channel      VARCHAR(100) DEFAULT 'direct',
            user_agent   TEXT,
            country      VARCHAR(100),
            last_seen    DATETIME     NOT NULL DEFAULT CURRENT_TIMESTAMP,
            created_at   DATETIME     NOT NULL DEFAULT CURRENT_TIMESTAMP,
            INDEX idx_session  (session_id),
            INDEX idx_last_seen (last_seen)
        ) $charset;";

        /* Push Subscriptions */
        $sql[] = "CREATE TABLE IF NOT EXISTS {$wpdb->prefix}smp_push_subscriptions (
            id           BIGINT UNSIGNED AUTO_INCREMENT PRIMARY KEY,
            endpoint     TEXT         NOT NULL,
            p256dh       TEXT,
            auth         TEXT,
            platform     VARCHAR(50),
            user_agent   TEXT,
            created_at   DATETIME NOT NULL DEFAULT CURRENT_TIMESTAMP,
            UNIQUE KEY uq_endpoint (endpoint(255))
        ) $charset;";

        /* לוג התראות */
        $sql[] = "CREATE TABLE IF NOT EXISTS {$wpdb->prefix}smp_notification_log (
            id           BIGINT UNSIGNED AUTO_INCREMENT PRIMARY KEY,
            order_id     BIGINT UNSIGNED,
            type         VARCHAR(50)  DEFAULT 'sale',
            message      TEXT,
            sent_at      DATETIME NOT NULL DEFAULT CURRENT_TIMESTAMP,
            success      TINYINT(1) DEFAULT 1,
            INDEX idx_sent_at (sent_at)
        ) $charset;";

        require_once ABSPATH . 'wp-admin/includes/upgrade.php';
        foreach ( $sql as $query ) dbDelta( $query );
    }

    public static function uninstall() {
        // שמור נתונים - לא נמחק בdea-activation
    }

    /* ─── Visitors Queries ─── */

    public static function upsert_visitor( $data ) {
        global $wpdb;
        $table = $wpdb->prefix . 'smp_visitors';
        $exists = $wpdb->get_var( $wpdb->prepare(
            "SELECT id FROM $table WHERE session_id = %s", $data['session_id']
        ) );
        if ( $exists ) {
            $wpdb->update( $table, [
                'page_url'   => $data['page_url']   ?? '',
                'page_title' => $data['page_title'] ?? '',
                'last_seen'  => current_time( 'mysql' ),
            ], [ 'session_id' => $data['session_id'] ] );
        } else {
            $wpdb->insert( $table, array_merge( $data, [
                'last_seen'  => current_time( 'mysql' ),
                'created_at' => current_time( 'mysql' ),
            ] ) );
        }
    }

    public static function get_active_visitors( $minutes = 5 ) {
        global $wpdb;
        $table = $wpdb->prefix . 'smp_visitors';
        return $wpdb->get_results( $wpdb->prepare(
            "SELECT * FROM $table WHERE last_seen >= DATE_SUB(NOW(), INTERVAL %d MINUTE) ORDER BY last_seen DESC",
            $minutes
        ) );
    }

    public static function get_channel_stats( $minutes = 5 ) {
        global $wpdb;
        $table = $wpdb->prefix . 'smp_visitors';
        return $wpdb->get_results( $wpdb->prepare(
            "SELECT channel, COUNT(*) as cnt FROM $table
             WHERE last_seen >= DATE_SUB(NOW(), INTERVAL %d MINUTE)
             GROUP BY channel ORDER BY cnt DESC",
            $minutes
        ) );
    }

    /* ─── Sales Queries ─── */

    public static function get_sales_by_day( $date ) {
        global $wpdb;
        $result = $wpdb->get_row( $wpdb->prepare(
            "SELECT 
                COUNT(*) as total_orders,
                SUM(meta_value) as total_revenue
             FROM {$wpdb->posts} p
             JOIN {$wpdb->postmeta} pm ON p.ID = pm.post_id
             WHERE p.post_type    = 'shop_order'
               AND p.post_status  IN ('wc-completed','wc-processing')
               AND pm.meta_key    = '_order_total'
               AND DATE(p.post_date) = %s",
            $date
        ) );
        return $result;
    }

    public static function get_sales_by_month( $year, $month ) {
        global $wpdb;
        return $wpdb->get_results( $wpdb->prepare(
            "SELECT 
                DATE(p.post_date) as day,
                COUNT(*) as orders,
                SUM(pm.meta_value) as revenue
             FROM {$wpdb->posts} p
             JOIN {$wpdb->postmeta} pm ON p.ID = pm.post_id
             WHERE p.post_type   = 'shop_order'
               AND p.post_status IN ('wc-completed','wc-processing')
               AND pm.meta_key   = '_order_total'
               AND YEAR(p.post_date)  = %d
               AND MONTH(p.post_date) = %d
             GROUP BY DATE(p.post_date)
             ORDER BY day ASC",
            $year, $month
        ) );
    }

    public static function get_today_summary() {
        return self::get_sales_by_day( date('Y-m-d') );
    }

    public static function get_month_summary( $year = null, $month = null ) {
        $year  = $year  ?? date('Y');
        $month = $month ?? date('n');
        $days  = self::get_sales_by_month( $year, $month );
        $total_orders  = 0;
        $total_revenue = 0;
        foreach ( $days as $d ) {
            $total_orders  += $d->orders;
            $total_revenue += $d->revenue;
        }
        return [
            'total_orders'  => $total_orders,
            'total_revenue' => $total_revenue,
            'days'          => $days,
        ];
    }

    /* ─── Push Subscriptions ─── */

    public static function save_subscription( $endpoint, $p256dh, $auth, $ua = '' ) {
        global $wpdb;
        $wpdb->replace( $wpdb->prefix . 'smp_push_subscriptions', [
            'endpoint'   => $endpoint,
            'p256dh'     => $p256dh,
            'auth'       => $auth,
            'user_agent' => $ua,
            'created_at' => current_time('mysql'),
        ] );
    }

    public static function get_all_subscriptions() {
        global $wpdb;
        return $wpdb->get_results( "SELECT * FROM {$wpdb->prefix}smp_push_subscriptions" );
    }

    public static function log_notification( $order_id, $message, $success = true ) {
        global $wpdb;
        $wpdb->insert( $wpdb->prefix . 'smp_notification_log', [
            'order_id' => $order_id,
            'message'  => $message,
            'success'  => $success ? 1 : 0,
            'sent_at'  => current_time('mysql'),
        ] );
    }
}
