<?php
if ( ! defined( 'ABSPATH' ) ) exit;

/**
 * SMP_Push_Notifications
 *
 * תומך בשתי שיטות:
 *  1. OneSignal (מומלץ) - REST API פשוט, בחינם עד 10,000 subscribers
 *  2. Web Push ידני (VAPID) - דורש Composer + minishlink/web-push
 */
class SMP_Push_Notifications {

    private $settings;

    public function __construct() {
        $this->settings = get_option( 'smp_settings', [] );
        add_action( 'wp_ajax_smp_save_push_subscription',    [ $this, 'ajax_save_subscription' ] );
        add_action( 'wp_ajax_nopriv_smp_save_push_subscription', [ $this, 'ajax_save_subscription' ] );
        add_action( 'smp_send_sale_notification',            [ $this, 'send_sale_notification' ] );
    }

    /* ─── AJAX: שמור subscription מהדפדפן ─── */
    public function ajax_save_subscription() {
        check_ajax_referer( 'smp_tracker', 'nonce' );
        $endpoint = sanitize_text_field( wp_unslash( $_POST['endpoint'] ?? '' ) );
        $p256dh   = sanitize_text_field( wp_unslash( $_POST['p256dh']   ?? '' ) );
        $auth     = sanitize_text_field( wp_unslash( $_POST['auth']     ?? '' ) );
        if ( $endpoint ) {
            SMP_Database::save_subscription( $endpoint, $p256dh, $auth, $_SERVER['HTTP_USER_AGENT'] ?? '' );
        }
        wp_send_json_success();
    }

    /* ─── שלח התראה למכירה ─── */
    public function send_sale_notification( $order_id ) {
        $order = wc_get_order( $order_id );
        if ( ! $order ) return;

        $total    = wc_price( $order->get_total() );
        $products = implode( ', ', array_map( fn($i) => $i->get_name(), $order->get_items() ) );
        $title    = '💰 מכירה חדשה!';
        $body     = "הזמנה #{$order_id} | {$order->get_total()} ₪\n{$products}";
        $url      = admin_url( "post.php?post={$order_id}&action=edit" );

        $method = $this->settings['push_method'] ?? 'onesignal';

        if ( $method === 'onesignal' ) {
            $this->send_via_onesignal( $title, $body, $url, $order_id );
        } else {
            $this->send_via_webpush( $title, $body, $url, $order_id );
        }
    }

    /* ─── OneSignal ─── */
    private function send_via_onesignal( $title, $body, $url, $order_id ) {
        $app_id  = $this->settings['onesignal_app_id']  ?? '';
        $api_key = $this->settings['onesignal_api_key'] ?? '';

        if ( ! $app_id || ! $api_key ) {
            SMP_Database::log_notification( $order_id, 'OneSignal: חסרים app_id / api_key', false );
            return;
        }

        $payload = [
            'app_id'            => $app_id,
            'included_segments' => [ 'Subscribed Users' ],
            'filters'           => [
                [ 'field' => 'tag', 'key' => 'role', 'relation' => '=', 'value' => 'admin' ]
            ],
            'headings'          => [ 'en' => $title, 'he' => $title ],
            'contents'          => [ 'en' => $body,  'he' => $body  ],
            'url'               => $url,
            'chrome_web_icon'   => SMP_PLUGIN_URL . 'assets/img/money-icon.png',
            'data'              => [ 'sound' => 'cash_register', 'order_id' => $order_id ],
            'android_sound'     => 'cash_register',
            'ios_sound'         => 'cash_register.wav',
        ];

        $response = wp_remote_post( 'https://onesignal.com/api/v1/notifications', [
            'headers' => [
                'Content-Type'  => 'application/json',
                'Authorization' => "Basic {$api_key}",
            ],
            'body'    => wp_json_encode( $payload ),
            'timeout' => 10,
        ] );

        $success = ! is_wp_error( $response ) && wp_remote_retrieve_response_code( $response ) === 200;
        SMP_Database::log_notification( $order_id, $body, $success );
    }

    /* ─── Web Push ידני (VAPID) ─── */
    private function send_via_webpush( $title, $body, $url, $order_id ) {
        $subscriptions = SMP_Database::get_all_subscriptions();
        if ( empty( $subscriptions ) ) return;

        $vapid_public  = $this->settings['vapid_public']  ?? '';
        $vapid_private = $this->settings['vapid_private'] ?? '';

        if ( ! $vapid_public || ! $vapid_private ) {
            SMP_Database::log_notification( $order_id, 'Web Push: חסרים VAPID keys', false );
            return;
        }

        $payload = wp_json_encode( [
            'title' => $title,
            'body'  => $body,
            'icon'  => SMP_PLUGIN_URL . 'assets/img/money-icon.png',
            'url'   => $url,
            'sound' => 'cash',
        ] );

        // שליחה לכל subscriber (ללא Composer - HTTP ידני)
        foreach ( $subscriptions as $sub ) {
            $this->send_webpush_message( $sub->endpoint, $sub->p256dh, $sub->auth, $payload, $vapid_public, $vapid_private );
        }
        SMP_Database::log_notification( $order_id, $body, true );
    }

    private function send_webpush_message( $endpoint, $p256dh, $auth, $payload, $vapid_public, $vapid_private ) {
        // הערה: מימוש מלא של VAPID דורש Composer + minishlink/web-push
        // להשתמש ב-OneSignal לפרודקשן - הרבה יותר פשוט ואמין
        error_log( "SMP Web Push to: $endpoint" );
    }

    /* ─── VAPID Key Generator ─── */
    public static function generate_vapid_keys() {
        if ( function_exists( 'openssl_pkey_new' ) ) {
            $key = openssl_pkey_new( [
                'curve_name'       => 'prime256v1',
                'private_key_type' => OPENSSL_KEYTYPE_EC,
            ] );
            if ( $key ) {
                $details    = openssl_pkey_get_details( $key );
                $public_key = base64_encode( $details['ec']['pub'] ?? '' );
                openssl_pkey_export( $key, $private_key_pem );
                return [
                    'public'  => $public_key,
                    'private' => $private_key_pem,
                ];
            }
        }
        return null;
    }
}
