<?php
if ( ! defined( 'ABSPATH' ) ) exit;

class SMP_Rest_API {

    public function __construct() {
        add_action( 'rest_api_init', [ $this, 'register_routes' ] );
    }

    public function register_routes() {
        $ns = 'smp/v1';

        // מכירות - סיכום היום
        register_rest_route( $ns, '/sales/today', [
            'methods'             => 'GET',
            'callback'            => [ $this, 'get_today_sales' ],
            'permission_callback' => [ $this, 'admin_permission' ],
        ] );

        // מכירות - לפי יום
        register_rest_route( $ns, '/sales/day/(?P<date>\d{4}-\d{2}-\d{2})', [
            'methods'             => 'GET',
            'callback'            => [ $this, 'get_day_sales' ],
            'permission_callback' => [ $this, 'admin_permission' ],
        ] );

        // מכירות - חודש
        register_rest_route( $ns, '/sales/month', [
            'methods'             => 'GET',
            'callback'            => [ $this, 'get_month_sales' ],
            'permission_callback' => [ $this, 'admin_permission' ],
        ] );

        // גולשים פעילים
        register_rest_route( $ns, '/visitors/live', [
            'methods'             => 'GET',
            'callback'            => [ $this, 'get_live_visitors' ],
            'permission_callback' => [ $this, 'admin_permission' ],
        ] );

        // מכירות אחרונות (לדאשבורד)
        register_rest_route( $ns, '/sales/recent', [
            'methods'             => 'GET',
            'callback'            => [ $this, 'get_recent_sales' ],
            'permission_callback' => [ $this, 'admin_permission' ],
        ] );
    }

    public function admin_permission() {
        return current_user_can( 'manage_woocommerce' );
    }

    public function get_today_sales( $request ) {
        $cache = get_transient( 'smp_today_stats' );
        if ( $cache ) return rest_ensure_response( $cache );

        $data = SMP_Database::get_today_summary();
        $result = [
            'orders'  => (int)   ( $data->total_orders  ?? 0 ),
            'revenue' => (float) ( $data->total_revenue ?? 0 ),
            'date'    => date('Y-m-d'),
        ];
        set_transient( 'smp_today_stats', $result, 60 );
        return rest_ensure_response( $result );
    }

    public function get_day_sales( $request ) {
        $date = $request['date'];
        $data = SMP_Database::get_sales_by_day( $date );
        return rest_ensure_response( [
            'orders'  => (int)   ( $data->total_orders  ?? 0 ),
            'revenue' => (float) ( $data->total_revenue ?? 0 ),
            'date'    => $date,
        ] );
    }

    public function get_month_sales( $request ) {
        $year  = (int) ( $request->get_param('year')  ?? date('Y') );
        $month = (int) ( $request->get_param('month') ?? date('n') );

        $cache_key = "smp_month_stats_{$year}_{$month}";
        $cache = get_transient( $cache_key );
        if ( $cache ) return rest_ensure_response( $cache );

        $data = SMP_Database::get_month_summary( $year, $month );
        set_transient( $cache_key, $data, 300 );
        return rest_ensure_response( $data );
    }

    public function get_live_visitors( $request ) {
        $visitors = SMP_Database::get_active_visitors( 5 );
        $channels = SMP_Database::get_channel_stats( 5 );

        return rest_ensure_response( [
            'count'    => count( $visitors ),
            'visitors' => $visitors,
            'channels' => $channels,
        ] );
    }

    public function get_recent_sales( $request ) {
        global $wpdb;
        $limit = (int) ( $request->get_param('limit') ?? 10 );

        $orders = $wpdb->get_results( $wpdb->prepare(
            "SELECT p.ID as order_id, p.post_date, pm.meta_value as total,
                    pm2.meta_value as billing_first, pm3.meta_value as billing_last
             FROM {$wpdb->posts} p
             LEFT JOIN {$wpdb->postmeta} pm  ON p.ID = pm.post_id  AND pm.meta_key  = '_order_total'
             LEFT JOIN {$wpdb->postmeta} pm2 ON p.ID = pm2.post_id AND pm2.meta_key = '_billing_first_name'
             LEFT JOIN {$wpdb->postmeta} pm3 ON p.ID = pm3.post_id AND pm3.meta_key = '_billing_last_name'
             WHERE p.post_type   = 'shop_order'
               AND p.post_status IN ('wc-completed','wc-processing')
             ORDER BY p.post_date DESC
             LIMIT %d",
            $limit
        ) );

        return rest_ensure_response( $orders );
    }
}
