<?php
if ( ! defined( 'ABSPATH' ) ) exit;

class SMP_Visitors {

    public function __construct() {
        add_action( 'wp_ajax_smp_track_visitor',        [ $this, 'ajax_track' ] );
        add_action( 'wp_ajax_nopriv_smp_track_visitor', [ $this, 'ajax_track' ] );
        add_action( 'wp_ajax_smp_heartbeat',            [ $this, 'ajax_heartbeat' ] );
        add_action( 'wp_ajax_nopriv_smp_heartbeat',     [ $this, 'ajax_heartbeat' ] );
    }

    /* ─── AJAX: Track pageview ─── */
    public function ajax_track() {
        check_ajax_referer( 'smp_tracker', 'nonce' );

        $session_id = sanitize_text_field( wp_unslash( $_POST['session_id'] ?? '' ) );
        $page_url   = esc_url_raw( wp_unslash( $_POST['page_url']   ?? '' ) );
        $page_title = sanitize_text_field( wp_unslash( $_POST['page_title'] ?? '' ) );
        $referrer   = esc_url_raw( wp_unslash( $_POST['referrer']   ?? '' ) );

        if ( ! $session_id ) { wp_send_json_error(); return; }

        $ip      = $this->get_client_ip();
        $channel = $this->detect_channel( $referrer );

        SMP_Database::upsert_visitor( [
            'session_id' => $session_id,
            'ip'         => $ip,
            'page_url'   => $page_url,
            'page_title' => $page_title,
            'referrer'   => $referrer,
            'channel'    => $channel,
            'user_agent' => substr( $_SERVER['HTTP_USER_AGENT'] ?? '', 0, 500 ),
        ] );

        wp_send_json_success( [ 'channel' => $channel ] );
    }

    /* ─── AJAX: Heartbeat - עדכן last_seen ─── */
    public function ajax_heartbeat() {
        check_ajax_referer( 'smp_tracker', 'nonce' );
        $session_id = sanitize_text_field( wp_unslash( $_POST['session_id'] ?? '' ) );
        if ( ! $session_id ) { wp_send_json_error(); return; }

        global $wpdb;
        $wpdb->update(
            $wpdb->prefix . 'smp_visitors',
            [ 'last_seen' => current_time('mysql') ],
            [ 'session_id' => $session_id ]
        );
        wp_send_json_success();
    }

    /* ─── Channel Detection ─── */
    private function detect_channel( $referrer ) {
        if ( empty( $referrer ) ) return 'ישיר';

        $referrer_lower = strtolower( $referrer );

        // Paid - UTM
        if ( isset( $_GET['utm_medium'] ) ) {
            $medium = strtolower( sanitize_text_field( $_GET['utm_medium'] ) );
            if ( $medium === 'cpc' || $medium === 'paid' )    return 'Google Ads';
            if ( $medium === 'social' )                        return 'Social Paid';
            if ( $medium === 'email' )                         return 'אימייל';
        }

        // Organic Search
        $search_engines = [ 'google.', 'bing.', 'yahoo.', 'yandex.', 'duckduckgo.' ];
        foreach ( $search_engines as $se ) {
            if ( strpos( $referrer_lower, $se ) !== false ) return 'חיפוש אורגני';
        }

        // Social Media
        $social = [
            'facebook'  => 'Facebook',
            'instagram' => 'Instagram',
            'tiktok'    => 'TikTok',
            'twitter'   => 'X / Twitter',
            'linkedin'  => 'LinkedIn',
            'youtube'   => 'YouTube',
            'pinterest' => 'Pinterest',
            'whatsapp'  => 'WhatsApp',
            'telegram'  => 'Telegram',
        ];
        foreach ( $social as $domain => $label ) {
            if ( strpos( $referrer_lower, $domain ) !== false ) return $label;
        }

        // Email clients
        if ( strpos( $referrer_lower, 'mail.' ) !== false ) return 'אימייל';

        return 'הפניה חיצונית';
    }

    /* ─── IP Detection ─── */
    private function get_client_ip() {
        foreach ( [ 'HTTP_CF_CONNECTING_IP', 'HTTP_X_FORWARDED_FOR', 'REMOTE_ADDR' ] as $key ) {
            if ( ! empty( $_SERVER[ $key ] ) ) {
                $ip = explode( ',', $_SERVER[ $key ] )[0];
                return sanitize_text_field( trim( $ip ) );
            }
        }
        return '0.0.0.0';
    }
}
